const { exec } = require('child_process');
const { promisify } = require('util');
const { existsSync } = require('fs');
const path = require('path');
const logger = require('../logger');
const { findFFprobe } = require('../ffprobe');

const execAsync = promisify(exec);

/**
 * Read cue points (chapter markers) from an MP4 file using bundled ffprobe
 * @param {string} videoPath - Path to the MP4 video file
 * @returns {Promise<Array>} Array of cue point objects with {startTime, endTime, title}
 */
async function readCuePoints(videoPath) {
  try {
    // Validate file exists
    if (!videoPath || !existsSync(videoPath)) {
      return [];
    }

    // Check if file is MP4 (cue points are only in MP4 format)
    const ext = path.extname(videoPath).toLowerCase();
    if (ext !== '.mp4' && ext !== '.m4v') {
      return [];
    }

    // Find bundled ffprobe binary
    const ffprobePath = findFFprobe();

    // Escape path for shell execution (handle spaces and special characters)
    const escapedPath = JSON.stringify(videoPath);


    // Execute ffprobe to get chapters
    const command = `"${ffprobePath}" -v quiet -print_format json -show_chapters ${escapedPath}`;

    let stdout, stderr;
    try {
      const result = await execAsync(command);
      stdout = result.stdout;
      stderr = result.stderr;
    } catch (error) {
      // execAsync throws on non-zero exit code, but may still have output
      stdout = error.stdout || '';
      stderr = error.stderr || '';
      
      // If file is invalid or ffprobe failed, return empty array
      if (error.code && error.code !== 0) {
        return [];
      }
    }

    if (stderr && stderr.trim()) {
      logger.warn('[CuePoints] FFprobe warnings:', stderr);
    }

    if (!stdout || !stdout.trim()) {
      return [];
    }

    // Parse JSON output
    let info;
    try {
      info = JSON.parse(stdout);
    } catch (parseError) {
      return [];
    }

    if (!info.chapters || !Array.isArray(info.chapters) || info.chapters.length === 0) {
      // FFprobe didn't find chapters, try manual parser as fallback
      const { readCuePointsWithParser } = require('./mp4-parser');
      try {
        const manualCuePoints = await readCuePointsWithParser(videoPath);
        if (manualCuePoints && manualCuePoints.length > 0) {
          logger.log(`[CuePoints] Found ${manualCuePoints.length} cue points from XMP metadata`);
          return manualCuePoints;
        }
      } catch (parserError) {
        // Silent fail
      }
      return [];
    }

    // Map chapters to cue points format
    const cuePoints = info.chapters.map((chapter, index) => {
      const startTime = parseFloat(chapter.start_time) || 0;
      const endTime = chapter.end_time ? parseFloat(chapter.end_time) : null;
      const title = chapter.tags?.title || chapter.tags?.TITLE || `Chapter ${index + 1}`;

      return {
        startTime,
        endTime,
        title
      };
    });

    if (cuePoints.length > 0) {
      logger.log(`[CuePoints] Found ${cuePoints.length} cue points from: ${path.basename(videoPath)}`);
    }
    return cuePoints;
  } catch (error) {
    // Log error but return empty array to not break the editor
    logger.error('Error reading cue points:', error.message || error);
    return [];
  }
}

module.exports = {
  readCuePoints
};
